#!/bin/bash
# 必须以 root 权限执行此脚本
if [ "$(id -u)" != "0" ]; then
  echo "This script must be run as root"
  exit 1
fi

# 定义配置文件路径
SCADA_ENV_FILE="/etc/xvisual_env"

# 定义 tar.gz 文件路径和临时目录
TAR_FILE="Xvisual.tar.gz"
TEMP_DIR="/tmp/Xvisual_extract"
INST_DIR="/opt/xvisual"
BACKUP_DIR="$TEMP_DIR/backupfiles"
DAEMONS_DIR="$TEMP_DIR/daemons"

# 定义 backup.list 文件路径
BACKUP_LIST="$TEMP_DIR/backup.list"

# 定义 SCADA 服务名称列表
SERVICES=(
  "xvisual.agent.service"
  "xvisual.server.service"
  "xvisual.comm.service"
  "xvisual.web.service"
)

# 设置 KEEP_CONFIG 默认值为 false
KEEP_CONFIG=false

# 检查并读取 Xvisual_WEB_PORT 环境变量
if [ -f "$SCADA_ENV_FILE" ]; then
  source "$SCADA_ENV_FILE"
fi

# 检查是否已经安装过
PREVIOUS_INSTALLATION_FOUND=false
if [ -d "$INST_DIR" ] && [ "$(ls -A "$INST_DIR")" ]; then
  echo "Found existing installation in $INST_DIR."
  PREVIOUS_INSTALLATION_FOUND=true
else
  echo "New installation start."
fi

# 如果已有安装，提示用户选择操作
if [ "$PREVIOUS_INSTALLATION_FOUND" = true ]; then
  while true; do
    read -p "Do you want to keep the existing configuration? (Y/N): " KEEP_RESPONSE
    case "$KEEP_RESPONSE" in
    [Yy]*)
      KEEP_CONFIG=true
      echo "Keeping existing configuration."
      break
      ;;
    [Nn]*)
      KEEP_CONFIG=false
      echo "Existing configuration will be overwritten."
      break
      ;;
    *)
      echo "Invalid input. Please enter Y or N."
      ;;
    esac
  done
else
  echo "Proceeding with a fresh installation."
fi

# 如果选择新安装或 Xvisual_WEB_PORT 未设置，则提示用户输入
if [ "$KEEP_CONFIG" = false ] || [ -z "$Xvisual_WEB_PORT" ]; then
  while true; do
    read -p "Enter a valid port number for Web (default is 80): " INPUT_PORT
    if [[ -z "$INPUT_PORT" ]]; then
      Xvisual_WEB_PORT=80
      break
    elif [[ "$INPUT_PORT" =~ ^[0-9]+$ ]] && [ "$INPUT_PORT" -ge 1 ] && [ "$INPUT_PORT" -le 65535 ]; then
      Xvisual_WEB_PORT="$INPUT_PORT"
      break
    else
      echo "Invalid port. Please enter a number between 1 and 65535."
    fi
  done

  # 将 Xvisual_WEB_PORT 写入配置文件
  echo "Xvisual_WEB_PORT=$Xvisual_WEB_PORT" > "$SCADA_ENV_FILE"
  echo "Xvisual_WEB_PORT saved to $SCADA_ENV_FILE."
else
  echo "Using existing Xvisual_WEB_PORT=$Xvisual_WEB_PORT from $SCADA_ENV_FILE."
fi

# 停止 SCADA 服务
echo "Stopping Xvisual services if running..."
for SERVICE in "${SERVICES[@]}"; do
  if systemctl is-active --quiet "$SERVICE"; then
    echo "Stopping $SERVICE..."
    systemctl stop "$SERVICE"
  else
    echo "$SERVICE is not running. Skipping..."
  fi
done

# 检查临时目录是否存在，如果存在则清空
if [ -d "$TEMP_DIR" ]; then
  echo "Cleaning existing temporary directory: $TEMP_DIR"
  rm -rf "$TEMP_DIR"/*
else
  echo "Creating temporary directory: $TEMP_DIR"
  mkdir -p "$TEMP_DIR"
fi

# 解压 tar.gz 文件到临时目录
echo "Extracting $TAR_FILE to $TEMP_DIR"
tar -xzf "$TAR_FILE" -C "$TEMP_DIR" --warning=no-unknown-keyword

# 进入临时目录
cd "$TEMP_DIR" || exit

# 使脚本可执行
chmod +x "$TEMP_DIR/install_dotnet.sh"
chmod +x "$TEMP_DIR/install_scada.sh"
chmod +x "$TEMP_DIR/restore.sh"
chmod +x "$TEMP_DIR/backup.sh"

# 如果 KEEP_CONFIG 为 true，则调用 backup.sh 备份用户数据
if [ "$KEEP_CONFIG" = true ]; then
  echo "Backing up existing configuration..."
  mkdir -p "$BACKUP_DIR"
  "$TEMP_DIR/backup.sh" "$INST_DIR" "$BACKUP_LIST" "$BACKUP_DIR"
  if [ $? -eq 0 ]; then
    echo "Backup completed successfully."
  else
    echo "Backup failed."
    exit 1
  fi
fi

# 使用绝对路径运行 install_dotnet.sh 脚本
echo "Running install_dotnet.sh script..."
"$TEMP_DIR/install_dotnet.sh"

if [ "$KEEP_CONFIG" = false ]; then
  LOG_DIR="/var/log/Xvisual"
  FSTAB_FILE="/etc/fstab"

  # 创建 RAM 驱动器
  echo "Creating RAM drive..."
  sudo mkdir -p "$LOG_DIR"

  # 备份并更新 /etc/fstab 文件
  echo "Backing up /etc/fstab..."
  sudo cp "$FSTAB_FILE" "${FSTAB_FILE}.bak"

  echo "Adding RAM drive configuration to /etc/fstab..."
  echo "tmpfs           $LOG_DIR  tmpfs   defaults,noatime,size=100m    0    0" | sudo tee -a "$FSTAB_FILE"
fi

# 运行 install_scada.sh 脚本并传入变量
"$TEMP_DIR/install_scada.sh" "$INST_DIR" "$TEMP_DIR"

# 将临时目录中的 daemons 文件夹内容复制到 /etc/systemd/system
if [ -d "$DAEMONS_DIR" ]; then
  cp -r "$DAEMONS_DIR/"* /etc/systemd/system/

  # 替换 xvisual.web.service 文件中的 {port} 为 Xvisual_WEB_PORT
  SCADA_WEB_SERVICE="/etc/systemd/system/xvisual.web.service"
  if [ -f "$SCADA_WEB_SERVICE" ]; then
    sed -i "s/{port}/$Xvisual_WEB_PORT/g" "$SCADA_WEB_SERVICE"
  else
    echo "Warning: $SCADA_WEB_SERVICE not found. Skipping port update."
  fi

else
  echo "No daemons directory found in $TEMP_DIR. Skipping daemon installation."
fi

# 如果 KEEP_CONFIG 为 true，则调用 restore.sh 还原用户数据
if [ "$KEEP_CONFIG" = true ]; then
  echo "Restoring configuration..."
  "$TEMP_DIR/restore.sh" "$INST_DIR" "$BACKUP_LIST" "$BACKUP_DIR"
  if [ $? -eq 0 ]; then
    echo "Restore completed successfully."
  else
    echo "Restore failed."
    exit 1
  fi
fi

echo "All installations completed."

# 启用 SCADA 服务
echo "Starting Xvisual services..."
for SERVICE in "${SERVICES[@]}"; do
  echo "Enabling $SERVICE..."
  systemctl enable "$SERVICE"
done

for SERVICE in "${SERVICES[@]}"; do
  echo "Starting $SERVICE..."
  systemctl start "$SERVICE"
done

echo "Xvisual setup completed successfully!"
