//
//  JXPrinterManager.h
//  JXPrinter
//
//  Created by 贾俊伟 on 2021/3/27.
//

#import <UIKit/UIKit.h>
#import "JXPrinterEnums.h"
#import "JXBleInfo.h"

NS_ASSUME_NONNULL_BEGIN

@interface JXPrinterManager : NSObject

/**
 * 打印服务单例对象（默认为蓝牙发送数据JXConnectTypeBle）
 * @return 打印服务单例对象
 */
+ (instancetype)printerManager;

/**
 * 打印服务单例对象
 * @param connectType 发送数据方式蓝牙或Wi-Fi
 * @return 打印服务单例对象
 */
+ (instancetype)printerManagerWithConnectType:(JXConnectType)connectType;

/**************************************************/
/**************   打印机设置指令  ********************/
/**************************************************/

/**
 * 读取蓝牙打印机状态
 * @param timeout 超时时间
 * @param success 成功回调
 * @param fail    超时时间到,失败回调
 */
- (void)readBlePrintStatus:(NSTimeInterval)timeout
                   success:(void (^)(PrintStatus blePrintStatus))success
                      fail:(void (^)(void))fail;

/**
 * 设置打印黑度(0-4)
 */
- (BOOL)setDarknessWithValue:(int)value;

/**
 * 设置打印质量(0-2)
 */
- (BOOL)setQualityWithValue:(int)value;

/**
 * 获取打印黑度(0-4), 返回nil表示获取失败
 */
- (NSString *)getDarkness;

/**
 * 获取打印质量(0-2), 返回nil表示获取失败
 */
- (NSString *)getQuality;

/**
 * 解锁
 */
- (BOOL)unlock;

/**
 * 打印自检页
 */
- (BOOL)printCheckPage;

/**************************************************/
/**************     CPCL打印    ********************/
/**************************************************/

- (BOOL)cpclPortSendCmd:(NSMutableString *)cmd;

/**
 * 设置打印纸张大小（打印区域）的大小，打印前必须先设置打印区域。（8个点为1毫米）
 * @param pageWidth 打印区域宽度
 * @param pageHeight 打印区域高度
 */
- (void)cpclPageSetup:(int)pageWidth pageHeight:(int)pageHeight;

/**
 * 页模式下打印
 * @param horizontal 0:正常打印，不旋转； 1：整个页面顺时针旋转180°后，再打印
 * @param skip 0：打印结束后不定位，直接停止； 1：打印结束后定位到标签分割线，如果无缝隙，最大进纸30mm后停止
 */
- (void)cpclPrint:(int)horizontal skip:(int)skip;

/**
 * 打印边框
 * @param lineWidth 边框线条宽度
 * @param topLeftX 矩形框左上角x坐标
 * @param topLeftY 矩形框左上角y坐标
 * @param bottomRightX 矩形框右下角x坐标
 * @param bottomRightY 矩形框右下角y坐标
 */
- (void)cpclDrawBoxWithLineWidth:(int)lineWidth
                        topLeftX:(int)topLeftX
                        topLeftY:(int)topLeftY
                    bottomRightX:(int)bottomRightX
                    bottomRightY:(int)bottomRightY;

/**
 * 打印线条
 * @param lineWidth 线条宽度
 * @param startX 线条起始点x坐标
 * @param startY 线条起始点y坐标
 * @param endX 线条结束点x坐标
 * @param endY 线条结束点y坐标
 */
- (void)cpclDrawLineWithLineWidth:(int)lineWidth
                           startX:(int)startX
                           startY:(int)startY
                             endX:(int)endX
                             endY:(int)endY;

/**
 * 页模式下打印文本
 * @param textX 起始横坐标
 * @param textY 起始纵坐标
 * @param text  打印的文本内容
 * @param fontSize 字体大小: 1：16点阵； 2：24点阵； 3：32点阵；4：24点阵放大一倍；5：32点阵放大一倍；6：24点阵放大两倍；7：32点阵放大两倍；其他：24点阵
 * @param rotate 旋转角度: 0：不旋转；    1：90度；    2：180°；    3:270°
 * @param bold 是否粗体 0：否； 1：是
 * @param reverse 是否反白 NO：不反白；YES：反白
 * @param underline  是有有下划线 NO:没有；YES：有
 */
- (void)cpclDrawTextWithTextX:(int)textX
                        textY:(int)textY
                         text:(NSString *)text
                     fontSize:(int)fontSize
                       rotate:(int)rotate
                         bold:(int)bold
                      reverse:(BOOL)reverse
                    underline:(BOOL)underline;

/**
 * 一维条码
 * @param startX 起始横坐标
 * @param startY 起始纵坐标
 * @param text    内容
 * @param type 条码类型： 0：CODE39; 1：CODE128；2：CODE93； 3：CODEBAR；4：EAN8； 5：EAN13；6：UPCA;  7:UPC-E; 8:ITF
 * @param rotate 0：水平显示; 1：竖直显示；
 * @param linewidth 条码线宽度
 * @param height 条码高度
 */
- (void)cpclDrawBarCodeWithStartX:(int)startX
                           startY:(int)startY
                             text:(NSString *)text
                             type:(int)type
                           rotate:(int)rotate
                        linewidth:(int)linewidth
                           height:(int)height;

/**
 * 二维码
 * @param startX 起始横坐标
 * @param startY 起始纵坐标
 * @param text    内容
 * @param rotate 旋转角度: 0：不旋转；    1：90度；    2：180°；    3:270°
 * @param ver  QrCode宽度(2-6)
 * @param lel  QrCode纠错等级(0-20)
 */
- (void)cpclDrawQrCodeWihtStartX:(int)startX
                          startY:(int)startY
                            text:(NSString *)text
                          rotate:(int)rotate
                             ver:(int)ver
                             lel:(int)lel;

/**
 * 图片
 * @param startX 起始横坐标
 * @param startY 起始纵坐标
 * @param image 图片
 */
- (void)cpclDrawGraphicWithStartX:(int)startX
                           startY:(int)startY
                            image:(UIImage *)image;

/**
 * 写入数据
 * @param cmd       要发送的cpcl指令
 */
- (void)cpclWriteCmd:(NSString *)cmd;

/**
 * 写入有返回数据的指令(返回打印机的回复结果，超时返回nil)
 * @param cmd 要发送的cpcl指令
 * @param timeout 查询结果超时时间（s）
 */
- (NSData *)cpclWriteWithResponseCmd:(NSString *)cmd
                             timeout:(NSInteger)timeout;

- (NSData *)cpclWriteWithResponseData:(NSData *)data
                              timeout:(NSInteger)timeout;

/**
 * 清除读取Buffer
 */
- (void)cpclFlushReadBuffer;


/**************************************************/
/***********    GLOBAL PAGE 模式打印   *************/
/**************************************************/

/**
 * 设置打印整体的宽度和高度
 * @param width  打印宽度
 * @param height 打印高度
 * @param paperType 纸张类型 0 间隙纸 1 连续纸 2 黑标
 */
- (void)globalPageSetUpWidth:(int)width
                      height:(int)height
                   paperType:(int)paperType;

/**
* 设置进入白色底绘图模式。
* @param isWhiteBG ; YES 进入白色底绘图模式，默认透明底绘图模式。
* 透明模式下，如果两个待打印图像重叠，打印效果是重叠图像。白色底模式下，如果两个待打印图像重叠，打印效果是后打图像遮盖新打图像
*/
- (void)globalSetWhiteBGMode:(BOOL)isWhiteBG;

/**
 * 打印
 * @param isUpsideDown YES, 旋转180打印
 * @param times        打印次数1-255
 */
- (void)globalPagePrint:(BOOL)isUpsideDown times:(int)times;

- (void)globalPagePrint;

/******************** 设置文本打印相关属性 设置后除非主动取消，否则一直生效 *******************/
// 行间距(单位点。1mm=8dot)
- (void)globalTextLineSpace:(int)lineSpace;
// 字间距(单位点。1mm=8dot)
- (void)globalTextLetterSpace:(int)letterSpace;
// 下划线
- (void)globalTextIsUnderline:(BOOL)isUnderline;
// 加粗
- (void)globalTextIsBold:(BOOL)isBold;
// 倾斜
- (void)globalTextIsItaly:(BOOL)isItaly;
// 反白
- (void)globalTextIsReverse:(BOOL)isReverse;
// 字体高度
- (void)globalTextFontSize:(int)fontSize;
// 字体宽度放大倍数
- (void)globalTextFontWidthScale:(int)fontWidthScale;
// 字体高度放大倍数
- (void)globalTextFontHeightScale:(int)fontHeightScale;
// 文本旋转角度
- (void)globalTextRotate:(RotationAngle)rotate;
// 文本是否换行
- (void)globalTextBreakLine:(BOOL)breakLine;

/**
 * 文本所有设置恢复默认值
 */
- (BOOL)globalTextReset;

/**
 * 打印文字
 *
 * @param startX    起始横坐标
 * @param startY    起始纵坐标
 * @param endX      结束横坐标
 * @param endY      结束纵坐标
 * @param content    打印的文本内容
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawTextWithStartX:(int)startX
                          startY:(int)startY
                            endX:(int)endX
                            endY:(int)endY
                         content:(NSString *)content;

/**
 * 打印边框
 *
 * @param startX   矩形框左上角x坐标
 * @param startY   矩形框左上角y坐标
 * @param width    矩形框宽度
 * @param height   矩形框高度
 * @param lineWidth 边框线条宽度
 * @param fullLine  YES:实线  NO: 虚线
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawBoxWithStartX:(int)startX
                         startY:(int)startY
                          width:(int)width
                         height:(int)height
                      lineWidth:(int)lineWidth
                       fullLine:(BOOL)fullLine;

/**
 * 打印线条
 * @param startX   线条起始点x坐标
 * @param startY   线条起始点y坐标
 * @param endX     线条结束点x坐标
 * @param endY     线条结束点y坐标
 * @param lineWidth 边框线条宽度
 * @param fullLine  YES:实线  NO: 虚线
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawLineWithStartX:(int)startX
                          startY:(int)startY
                            endX:(int)endX
                            endY:(int)endY
                       lineWidth:(int)lineWidth
                        fullLine:(BOOL) fullLine;
/**
 * 调用{drawBarCode: start_y: width: height: content: rotate: format:}使用，打印条码的文字，仅生效一次
 *
 * @param isPositionBelow 是否在条码下方展示
 * @param fontSize 字体大小
 * @param isBold  加粗
 * @param margin 和条码的间距
 * @param breakLine 是否截取
 */
- (void)globalSetBarcodeTextWithIsPositionBelow:(BOOL)isPositionBelow
                                       fontSize:(int)fontSize
                                         isBold:(BOOL)isBold
                                         margin:(int)margin
                                      breakLine:(BOOL)breakLine;

/**
 * 一维条码
 *
 * @param startX 一维码起始横坐标
 * @param startY 一维码起始纵坐标
 * @param content 内容，条码类型默认code128
 * @param rotate  旋转角度：0：不旋转；    1：90度；    2：180°；    3:270°
 * @param width   条码宽度
 * @param height  条码高度
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawBarCodeWithStartX:(int)startX
                             startY:(int)startY
                              width:(int)width
                             height:(int)height
                            content:(NSString *)content
                             rotate:(RotationAngle)rotate
                             format:(BarCodeFormat)format;

/**
 * 11.二维码
 *
 * @param startX 二维码起始横坐标
 * @param startY 二维码起始纵坐标
 * @param content 二维码内容
 * @param rotate  旋转角度：0：不旋转；    1：90度；    2：180°；    3:270°
 * @param width    二维码宽度
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawQrCodeWithStartX:(int)startX
                            startY:(int)startY
                             width:(int)width
                            height:(int)height
                           content:(NSString *)content
                            rotate:(RotationAngle)rotate
                            format:(QRCodeFormat)format;

/**
 * 12.图片
 *
 * @param startX 图片起始点x坐标
 * @param startY 图片起始点y坐标
 * @param image     图片
 * @return BOOL 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalDrawGraphicWithStartX:(int)startX startY:(int)startY image:(UIImage *)image;

/**
 * 向蓝牙设备写入二进制数据，自动进行分段发送
 * @param data       要发送的二进制数据
 */
- (void)globalWriteData:(NSData *)data;

/**
 * 写入有返回数据的指令(返回打印机的回复结果，超时返回nil)
 * @param data 要发送的二进制数据
 * @param timeout 查询结果超时时间（s）
 */
- (NSData *)globalWriteWithResponseData:(NSData *)data
                                timeout:(NSInteger)timeout;

/**
 * 清除读取Buffer
 */
- (void)globalFlushReadBuffer;

/**
  * 找标签缝，找不到走纸10cm
 */
- (void)globalFeed;

/**
  * 设置打印黑度(0-4)
 */
- (BOOL)globalDarknessWithValue:(int)value;

/**
  * 设置打印前退纸 (X35新增)
 设置打印单张标签前退纸的点数，在发送打印指令前设置
 */
- (BOOL)globalRetreatPaperWithValue:(int)value;

/**
  * 设置打印后进纸 (X35新增)
 设置打印单张标签后进纸的点数，在发送打印指令前设置
 当上一张纸的定位方式设置成5时，下一张标签打印前应设置W=0
 */
- (BOOL)globalFeedPaperWithValue:(int)value;

/**
  * 设置打印后进纸 (X35新增)
 设置打印单张标签后定位的方式，在发送打印指令前设置
 */
- (BOOL)globalFeedPaperWithStyle:(int)value;

/**
  * 打印机Feed方式预设值 (X35新增)
 value = 0：退纸打印机预设值
 value = 1：进纸打印机预设值
 */
- (BOOL)globalAutoFeedPaperWithStyle:(int)value;

/**
 * 控制马达行为指令
 *
 * @param cmdId      当 ID = 0：往前进纸 K个点，然后往后退P个点
 *                   当 ID = 1：往前进纸找标签缝处，如果没有找到标签缝，最多走K个点，否则找到标签缝后停止后再往前走P个点。
 *                   当 ID = 2：往后退K个点，停下来后往前走P个点
 *                   当 ID = 3：往前进纸找左黑标，如果没有找到黑标，最多走K个点，否则找到黑标后停止后再往前走P个点。
 *                   当 ID = 4：往前进纸找右黑标，如果没有找到黑标，最多走K个点，否则找到黑标后停止后再往前走P个点。
 * @param firstStep  K
 * @param secondStep P
 */
- (void)globalMotorControl:(int)cmdId fistStep:(int)firstStep secondStep:(int)secondStep;

/**
  * 设置打印质量(0-2)
 */
- (BOOL)globalQualityWithValue:(int)value;

/**
 * 获取打印机配置
 *
 * @param type      配置id
 */
- (NSString *)globalReadPrinterSettingInfoWithType:(JXPrinterInfoType)type;

/**
 * 修改打印机配置
 *
 * @param type   配置id
 * @param value  配置的值
 */
- (BOOL)globalSetPrinterSettingInfoWithType:(JXPrinterInfoType)type value:(NSString *)value;

/**
  * 保存打印机配置（下次开机仍然有效）
 */
- (BOOL)globalSaveSetting;


/**
  * 获取用户数据(打印机的关机次数、开盖次数、打印长度)
 */
- (JXPrinterUseInfo *)globalReadUseInfo;

/**
  * 获取固件版本号
 */
- (NSString *)globalReadFirmwareVersionInfo;

/**
  * 获取光耦数据
 */
- (JXPrinterOptocouplerInfo *)globalReadOptocouplerInfo;

/**
  * 获取打印机单字节字库个数(英文)
 */
- (NSNumber *)globalReadEnFontLibNumber;

/**
  * 获取打印机双字节字库个数(汉字)
 */
- (NSNumber *)globalReadZhFontLibNumber;

/**
  * 获取打印机单字节字库信息(英文)
 */
- (JXPrinterFontLibInfo *)globalReadEnFontLibInfoWithIndex:(NSInteger)index;

/**
  * 获取打印机双字节字库信息(汉字)
 */
- (JXPrinterFontLibInfo *)globalReadZhFontLibInfoWithIndex:(NSInteger)index;

/**
  * 获取打印机头片温度和电压
 */
- (JXPrinterHeadAndBatteryInfo *)globalReadHeadAndBatteryInfo;

/**
  * 进入蓝牙更新模式
 */
- (BOOL)globalEnterUpgradeMode;

/**
  * 确认是否进入蓝牙更新模式
 */
- (BOOL)globalConfirmEnterUpgradeMode;

/**
  * 检测固件/字库升级数据大小是否合法
 */
- (BOOL)globalCheckUpgradeDataWithDataLength:(int)dataLength isFontKit:(BOOL)isFontKit;

/**
  * 发送固件/字库升级数据
 */
- (BOOL)globalSendUpgradeDataWithData:(NSData *)data isFontKit:(BOOL)isFontKit;

/**
  * 升级重启打印机
 */
- (BOOL)globalPrinterRestart;


/**************************************************/
/**************      ESC打印    ********************/
/**************************************************/

/**
 * 1、初始化打印机。
 * 使所有设置恢复到打印机开机时的默认值模式。
 */
- (void)escReset;

/**
 * 2、设置打印文字字体属性
 */

/// 选择加粗模式
/// @param b b为YES时选择加粗模式，b为NO时取消加粗模式。
- (void)escBold:(BOOL)b;

/// 选择/取消下划线模式
/// @param n 当n=1或n=49时选择下划线模式且设置为1点宽，当n=2或n=50时选择下划线模式且设置为2点宽，当n取其他值时取消下划线模式。
- (void)escUnderline:(NSInteger)n;

/// 设置默认行高。 将行间距设为约 3.75mm{30/203"}。
- (void)escDefaultLineHeight;

/// 设置行高为[n×纵向或横向移动单位]英寸。
/// @param n n表示行高值。当n＜0时设置行高为0，当n＞255时设置行高为255[n×纵向或横向移动单位]英寸，           当0≤n≤255时设置行高为[n×纵向或横向移动单位]英寸。
- (void)escLineHeight:(NSInteger)n;

/// 设定右侧字符间距。
/// @param n 当n＜0时设定右侧字符间距为0，当n＞255时设定右侧字符间距为【255×（水平或垂直移动单位）】,           当0≤n≤255时设定右侧字符间距为【n×（水平或垂直移动单位）】。
- (void)escRightSpace:(NSInteger)n;

/// 选择字体
/// @param n 当n=1或n=49时选择字体B，当n=2或n=50时选择字体C，当n=3或n=51时选择字体D，当n为其他值时选择字体A。
- (void)escFont:(NSInteger)n;

/// 选择/取消顺时针旋转90°
/// @param n 当n=1或n=49时设置90°顺时针旋转模式，当n=2或n=50时设置180°顺时针旋转模式，当n=3或n=51时设置270°顺时针旋转模式，          当n取其他值时取消旋转模式。
- (void)escRotate:(NSInteger)n;

/// 设置绝对打印位置 将当前位置设置到距离行首（nL+nH×256）×（横向或纵向移动单位）处。当nL＜0或nL＞255时将nL设置为0，当nH＜0或nH＞255时将nH设置为0。
- (void)escAbsolutePrintPosition:(NSInteger)nL nH:(NSInteger)nH;

/// 设定相对打印位置 将打印位置从当前位置移至（nL+nH×256）×（水平或垂直运动单位）。当nL＜0时设置nL=0，当nL＞255时设置nL=255。 当nH＜0时设置nH=0，当nH＞255时设置nH=255。
- (void)escRelativePrintPosition:(NSInteger)nL nH:(NSInteger)nH;

/**
 * 3、设定左边距。
 * 当0≤nL≤255且0≤nH≤255时，将左边距设为【(nL+nH×256)×(水平移动单位)】。当nL和nH取其他值时将左边距设为0。
 */
- (void)escLeftMargin:(NSInteger)nL nH:(NSInteger)nH;

/**
 * 4、选择对齐模式。
 * @param n 当n=1或n=49时选择居中对齐，当n=2或n=50时选择右对齐，当n取其他值时选择左对齐。
 */
- (void)escAlign:(NSInteger)n;

/**
 * 5、水平制表符。
 * 将打印位置移动至下一水平制表符位置。
 */
- (void)escNextHorizontalTab;

/**
 * 6、设置水平制表符位置。
 * @param n n的长度表示横向跳格数，n[k]表示第k个跳格位置的值。当n的长度大于32时，只取前32个值；当n[k]大于等于n[k-1]时忽略该命令。
 *          当n[k]≤0或n[k]≥255时，忽略该命令。
 */
- (void)escHorizontalTabPosition:(NSArray *)n;

/**
 * 7、选择国际字符集。
 * @param n 当n≤0或n＞13时选择America字符集，当n=1时选择France字符集，当n=2时选择German字符集，当n=3时选择UK字符集，
 *          当n=4时选择Denmar字符集，当n=5时选择Sweden字符集，当n=6时选择Italy字符集，当n=7时选择Spain I字符集，当n=8时选择Japan字符集，
 *          当n=9时选择Norway字符集，当n=10时选择Denmar字符集，当n=11时选择Spain II字符集，当n=12时选择Latin字符集，当n=13时选择Korea字符集。
 */
- (void)escNationalCharacterSet:(NSInteger)n;

/**
 * 8、选择字符代码页。
 * @param n 当n=1时选择Page 1 Katakana，当n=2时选择Page 2 Multilingual(Latin-1) [CP850]，当n=3时选择Page 3 Portuguese [CP860]，
 *          当n=4时选择Page 4 Canadian-French [CP863]，当n=5时选择Page 5 Nordic [CP865]，当n=6时选择Page 6 Slavic(Latin-2) [CP852]，
 *          当n=7时选择Page 7 Turkish [CP857]，当n=8时选择Page 8 Greek [CP737]，当n=9时选择Page 9 Russian(Cyrillic) [CP866]，
 *          当n=10时选择Page 10 Hebrew [CP862]，当n=11时选择Page 11 Baltic [CP775]，当n=12时选择Page 12 Polish，
 *          当n=13时选择Page 13 Latin-9 [ISO8859-15]，当n=14时选择Page 14 Latin1[Win1252]，当n=15时选择Page 15 Multilingual Latin I + Euro[CP858]，
 *          当n=16时选择Page 16 Russian(Cyrillic)[CP855]，当n=17时选择Page 17 Russian(Cyrillic)[Win1251]，当n=18时选择Page 18 Central Europe[Win1250]，
 *          当n=19时选择Page 19 Greek[Win1253]，当n=20时选择Page 20 Turkish[Win1254]，当n=21时选择Page 21 Hebrew[Win1255]，
 *          当n=22时选择Page 22 Vietnam[Win1258]，当n=23时选择Page 23 Baltic[Win1257]，当n=24时选择Page 24 Azerbaijani，
 *          当n=30时选择Thai[CP874]Thai[CP874]，当n=40时选择Page 25 Arabic [CP720]，当n=41时选择Page 26 Arabic [Win 1256]，
 *          当n=42时选择Page 27 Arabic (Farsi)，当n=43时选择Page 28 Arabic presentation forms B，当n=50时选择Page 29 Page 25 Hindi_Devanagari，
 *          当n=252时选择Page 30 Japanese[CP932]，当n=253时选择Page 31 Korean[CP949]，当n=254时选择Page 32 Traditional Chinese[CP950]，
 *          当n=255时选择Page 33 Simplified Chinese[CP936]。
 *          当n取其他值时选择else if(n == 252) esc_character_code_page[2] = 0x01。
 */
- (void)escCharacterCodePage:(NSInteger)n;

/**
 * 9、设定汉字模式。
 * @param b 当b为YES时选择汉字模式，当b为NO时取消汉字模式。
 */
- (void)escChineseMode:(BOOL)b;

/**
 * 10、选择字符大小。
 * @param n 当n=2时2倍高，当n=3时3倍高，当n=4时4倍高，当n=20时2倍宽，当n=30时3倍宽，当n=40时4倍宽，当n=22时2倍宽高，当n=33时3倍宽高，
 *          当n=44时4倍宽高，当n取其他值时1倍宽高。
 */
- (void)escCharacterSize:(NSInteger)n;

/**
 * 11、选择/取消黑白反显打印模式。
 * @param b 当b为YES时选择黑白反显打印模式，当b为NO时取消黑白反显打印模式。
 */
- (void)escBlackWhiteReverse:(BOOL)b;

/**
 * 12、打印文本。
 * @param text 表示所要打印的文本内容。
 */
- (void)escPrintText:(NSString *)text;

/**
 * 13、设置参数打印条码。
 * @param HRIPosition HRI_position表示HRI字符打印位置(当HRI_position=1或HRI_position=49时HRI字符显示在条形码上方；
 *                     当HRI_position=2或HRI_position=50时HRI字符显示在条形码下方；当HRI_position取其他值时HRI字符不显示)。
 * @param HRIFont HRI_font表示HRI字符字体（当HRI_font=1或HRI_font=49时选择字体B，当HRI_font取其他值时选择字体A）。
 * @param width width表示条码宽度（当width=2时设置条形码宽度为2，当width=3时设置条形码宽度为3，当width取其他值时设置条形码宽度为1），
 * @param height height表示条码高度（当1<=height<=255时设置条码高度为height，当height取其他值时设置条码高度为162），
 * @param type type表示条码类型（当type=0或type=65时选择条码类型为UPC-A，当type=1或type=66时选择条码类型为UPC-E，
 *             当type=2或type=67时选择条码类型为EAN13，当type=3或type=68时选择条码类型为EAN8，当type=4或type=69时选择条码类型为CODE39，
 *             当type=5或type=70时选择条码类型为ITF，当type=6或type=71时选择条码类型为CODABAR，当type=7或type=72时选择条码类型为CODE93，
 *             当type=8或type=73时选择条码类型为CODE128），
 * @param content content表示条码内容（UPC-A（长度为11、12）、UPC-E（长度为7、8、11、12）、EAN13（长度为12、13）、EAN8（长度为7、8）、
 *                ITF（长度为大于2的偶数）只支持数字；
 *                CODE39（长度大于1且小于255，支持数字、英文、空格、‘$’、‘%’、‘*’、‘+’、‘-’、‘.’、‘/’）；
 *                CODE93（长度大于1且小于255，支持数字、英文、空格、‘$’、‘%’、‘+’、‘-’、‘.’、‘/’）；
 *                CODABAR（长度大于2且小于255，支持数字、英文ABCDabcd、‘$’、‘+’、‘-’、‘.’、‘/’、‘:’）；
 *                CODE128（长度大于2且小于255，支持所有英文）。
 */
- (void)escBarcodeWithHRIPosition:(NSInteger)HRIPosition
                          HRIFont:(NSInteger)HRIFont
                            width:(NSInteger)width
                           height:(NSInteger)height
                             type:(NSInteger)type
                          content:(NSString *)content;

/**
 * 14、打印二维码。
 * @param size 二维码放大倍数（0≤size≤3），
 * @param content content表示要打印的二维码内容。
 */
- (void)escPrintBarcode2dWithSize:(NSInteger)size
                          content:(NSString *)content;

/// 打印二维码。
/// @param size 二维码放大倍数（0≤size≤3）
/// @param qrVersion qr版本，1-10，取值越大打印条码宽度越大，推荐3
/// @param level qr纠错等级，0-3，值越高越容易识别，推荐1
/// @param content 二维码内容
- (void)escPrintBarcode2dWithSize:(NSInteger)size
                        qrVersion:(NSInteger)qrVersion
                            level:(NSInteger)level
                          content:(NSString *)content;

/**
 * 15、打印光栅位图
 */
- (BOOL)escRasterImage:(UIImage *)image;

/**
 * 16、选择位图模式打印图片。
 * @param m m表示位图模式。当m=1时位图模式为8点双密度，当m=32时位图模式为24点单密度，当m=33时位图模式为24点双密度，
 *          除m=1,32,33之外位图模式都为8点单密度。
 * @param image bitmap为要打印的位图。由于打印纸宽度有限，图片不可太大。
 */
- (void)escBitmapMode:(NSUInteger)m image:(UIImage *)image;

/**
 * 17、打印和行进。
 * 基于当前的行间距，打印缓冲区内的数据并走纸一行。
 */
- (void)escPrintFormfeed;

/**
 * 18、打印并向前走纸n行。
 * @param n 当n＜0时进纸0行，当n＞255时进纸255行，当0≤n≤255时进纸n行。
 */
- (void)escPrintFormfeedRow:(NSInteger)n;

/**
 * 19、打印并走纸到右黑标处。
 */
- (void)escPrintToRightBlackLabel;

/**
 * 20、打印并走纸到左黑标处。
 * 将打印缓冲区中的数据全部打印出来并走纸到左黑标处。
 */
- (void)escLeftBlackLabel;

/**
 * 21、走纸到标签处。
 */
- (BOOL)escPrintToLabel;

/**
 * 22、向蓝牙设备写入数据
 * @param data 要发送的二进制数据
 */
- (void)escWriteData:(NSData *)data;

/**
 * 23、写入有返回数据的指令(返回打印机的回复结果，超时返回nil)
 * @param data 要发送的二进制数据
 * @param timeout 查询结果超时时间（s）
 */
- (NSData *)escWriteWithResponseData:(NSData *)data
                             timeout:(NSInteger)timeout;

/**
 * 24、清除读取Buffer
 */
- (void)escFlushReadBuffer;

/**************************************************/
/***********    GLOBAL LINE 模式打印   *************/
/**************************************************/

/******************** 设置文本打印相关属性 设置后除非主动取消，否则一直生效 *******************/
// 行间距(单位点。1mm=8dot)
- (void)globalLineLineSpace:(int)lineSpace;
// 字间距(单位点。1mm=8dot)
- (void)globalLineLetterSpace:(int)letterSpace;
// 下划线
- (void)globalLineIsUnderline:(BOOL)isUnderline;
// 加粗
- (void)globalLineIsBold:(BOOL)isBold;
// 倾斜
- (void)globalLineIsItaly:(BOOL)isItaly;
// 反白
- (void)globalLineIsReverse:(BOOL)isReverse;
// 字体高度
- (void)globalLineFontSize:(int)fontSize;
// 字体宽度放大倍数
- (void)globalLineFontWidthScale:(int)fontWidthScale;
// 字体高度放大倍数
- (void)globalLineFontHeightScale:(int)fontHeightScale;

/**
 * 清除缓冲区数据，所有设置恢复默认值
 */
- (BOOL)globalLineReset;

/**
 * 设置左边距
 *
 * @param margin 0-65535，需要考虑左右边距，如果同时设置左右边距当可打印区域<100时设置失败
 * @return YES/NO
 */
- (BOOL)globalLineLeftMargin:(int)margin;

/**
 * 设置右边距
 *
 * @param margin 0-255，需要考虑左右边距，如果同时设置左右边距当可打印区域<100时设置失败
 * @return YES/NO
 */
- (BOOL)globalLineRightMargin:(int)margin;

/**
 * 设置相对打印位置
 *
 * @param align 1居中，2居右，其它居左
 * @return YES
 */
- (BOOL)globalLineAlign:(int)align;

/**
 * 设置打印机最大打印区域，必须设置，否则默认384
 */
- (BOOL)globalLineMaxPrintWidth:(int)maxWidth;

/**
 * 强制打印缓冲区
 */
- (BOOL)globalLinePrint;

/**
 * 走纸
 *
 * @param n n行
 */
- (BOOL)globalLineFeed:(int)n;

/**
 * 绘制文本
 */
- (BOOL)globalLineDrawText:(NSString *)content;

/**
 * 绘制位图
 */
- (BOOL)globalLineDrawGraphicWithImage:(UIImage *)image;

/**
 * 打印线条
 *
 * @param lineLength 线条长度，如果超出最大打印长度，只打印可打印部分，可调用globalLinePrint可强制打印缓冲区
 * @param lineWidth  线条宽度
 * @param isSolid    YES:实线  NO: 虚线
 * @return boolean 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalLineDrawLineWithLineLength:(int)lineLength
                               lineWidth:(int)lineWidth
                                 isSolid:(BOOL)isSolid;

/**
 * 调用{drawBarCodeWithWidth: height: content: rotate: format:}前使用，打印条码的文字，仅生效一次
 *
 * @param isPositionBelow 是否在条码下方展示
 * @param fontSize 字体大小
 * @param isBold  加粗
 * @param margin 和条码的间距
 * @param breakLine YES截断 NO换行
 */
- (void)globalLineSetBarcodeTextWithIsPositionBelow:(BOOL)isPositionBelow
                                           fontSize:(int)fontSize
                                             isBold:(BOOL)isBold
                                             margin:(int)margin
                                          breakLine:(BOOL)breakLine;

/**
 * 一维条码
 * 默认不显示文字。如果需要打印文字，需要先调用{setBarcodeTextWithIsPositionBelow:fontSize:isBold:margin:breakLine:}
 * 可调用{@link #print()}可强制打印缓冲区
 *
 * @param width   条码宽度
 * @param height  条码高度
 * @param content 内容
 * @param rotate  旋转角度
 * @return boolean 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalLineDrawBarCodeWithWidth:(int)width
                                height:(int)height
                               content:(NSString *)content
                                rotate:(RotationAngle)rotate
                                format:(BarCodeFormat)format;

/**
 * 二维码
 *
 * @param width   宽度
 * @param height  高度
 * @param content 内容
 * @param rotate  旋转角度
 * @return boolean 返回值：YES（发送成功） | NO （发送失败）
 */
- (BOOL)globalLineDrawQrCodeWithWidth:(int)width
                               height:(int)height
                              content:(NSString *)content
                               rotate:(RotationAngle)rotate
                               format:(QRCodeFormat)format;

/**
 * 打印曲线的线段（曲线是由连续的线段组成的，点可以理解为长度为1的线段）
 *
 * @param xPos 先算的起始点，从左侧0点开始，最大243
 *             X1-X2,X3-X4...X2n-1 - X2n打印多条线段
 *             如果仅打印一个点，X2=X1,即参数传 X,X
 * @returns YES/NO
 */
- (BOOL)globalLineDrawLineSegments:(NSArray<NSNumber *> *)xPos;

/**
 * 向蓝牙设备写入二进制数据，自动进行分段发送
 * @param data       要发送的二进制数据
 */
- (void)globalLineWriteData:(NSData *)data;

/**
 * 写入有返回数据的指令(返回打印机的回复结果，超时返回nil)
 * @param data 要发送的二进制数据
 * @param timeout 查询结果超时时间（s）
 */
- (NSData *)globalLineWriteWithResponseData:(NSData *)data
                                    timeout:(NSInteger)timeout;

/**
 * 清除读取Buffer
 */
- (void)globalLineFlushReadBuffer;

/**************************************************/
/***********    压缩图片 模式打印   *************/
/**************************************************/

/// 压缩图片打印 (YC方案新增) 压缩比ratio=0.95 流控分包最大值size=2304
/// @param image 图片
/// @param compress 是否使用压缩算法
/// @param fluid 是否流控（YES等结果返回再发下一包，NO不查结果， 全部发送）
/// @param success 发送成功状态(非流控直接返回成功,不查结果)
- (void)compressDrawGraphicWithImage:(UIImage *)image
                            compress:(BOOL)compress
                               fluid:(BOOL)fluid
                             success:(void (^)(BOOL success))success;

/// 压缩图片打印 (YC方案新增) 自定义ratio和size
/// @param image 图片
/// @param compress 是否使用压缩算法
/// @param ratio 压缩比 =压缩后/压缩前 <= 压缩比时使用压缩
/// @param fluid 是否流控（YES等结果返回再发下一包，NO不查结果， 全部发送）
/// @param size 流控发送分包最大值
/// @param success 发送成功状态(非流控直接返回成功,不查结果)
- (void)compressDrawGraphicWithImage:(UIImage *)image
                            compress:(BOOL)compress
                       CompressRatio:(CGFloat)ratio
                               fluid:(BOOL)fluid
                           ChunkSize:(int)size
                             success:(void (^)(BOOL success))success;

@end

NS_ASSUME_NONNULL_END

